!Likelihood code used in Polarbear et al 2019
!Poalrbear 650deg2  EE power spectrum
!For questions, please contact Anh Pham
!Code is based on the publicly released SPTpol code by Henning et al 2017.

  module CMB_Polarbear_EE
    use Likelihood_Cosmology
    use CosmoTheory
    use CosmologyTypes
    use FileUtils
    use MatrixUtils

    implicit none

    integer :: nall
    double precision, dimension(:,:), allocatable ::  cov, beam_err, cov_w_beam,cov_tmp
    double precision, dimension(:,:), allocatable :: windows, dl_derivative, raw_spectra
    integer :: pb_windows_lmin, pb_windows_lmax,n_beam_terms
    double precision, dimension(:), allocatable :: cl_to_dl_conversion,ells,rawspec_factor,deriv_factor, spec

    logical :: SuccessfulPBEEInitialization
    logical :: correct_aberration
    logical :: PB_Pcal_prior
    logical :: PB_kappa_prior
    logical :: PB_alphaEE_prior
    logical :: PB_dustEE_prior
    logical :: printDlPB
    logical :: binaryCov, binaryWindows, binaryBeamErr ! IO options.
    
    double precision :: meanPcal, sigmaPcal !in power-spectrum-space
    double precision :: meankappa, sigmakappa
    double precision :: meanAlphaEE, sigmaAlphaEE
    double precision :: meanDustEE, sigmaDustEE

    Type, extends(TCMBLikelihood) :: TPolarbearEELike
  contains
    procedure :: ReadIni => PB_EE_ReadIni
    procedure :: InitPBData
    procedure :: LogLike => PBEELnLike
 end Type TPolarbearEELike

contains

 subroutine PB_EE_ReadIni(this, Ini)
   !use IniFile
   !use IniObjects
   implicit none
   class(TPolarbearEELike) :: this
   class(TSettingIni) :: Ini
   character (LEN=Ini_Enumeration_Len) :: desc_file
   character (LEN=Ini_Enumeration_Len) :: bp_file, param_file
   character (LEN=Ini_Enumeration_Len) :: cov_file,beamerr_file
   character (LEN=Ini_Enumeration_Len) :: window_folder

   correct_aberration = Ini%Read_Logical('correct_aberration', .false.)


   PB_Pcal_prior = Ini%Read_Logical('pb_pcal_prior', .false.)
   meanPcal = Ini%Read_Real('pb_meanPcal', 1.0)
   sigmaPcal = Ini%Read_Real('pb_sigmaPcal', 0.02)
   sigmaPcal = log(1+sigmaPcal)

   PB_kappa_prior = Ini%Read_Logical('pb_kappa_prior', .false.)
   meankappa = Ini%Read_Real('pb_meankappa', 0.0)
   sigmakappa = Ini%Read_Real('pb_sigmakappa', 0.001)

   PB_alphaEE_prior = Ini%Read_Logical('pb_alphaEE_prior', .false.)
   meanAlphaEE = Ini%Read_Real('pb_meanAlphaEE', -0.58)
   sigmaAlphaEE = Ini%Read_Real('pb_sigmaAlphaEE', 0.02)

   PB_dustEE_prior = Ini%Read_Logical('pb_dustEE_prior', .false.)
   meanDustEE = Ini%Read_Real('pb_meanDustEE', 0.0094)
   sigmaDustEE = Ini%Read_Real('pb_sigmaDustEE', 0.002)

   if (feedback > 1) then
      print *, 'PB priors:'

      print *, 'Use Pcal prior: ', PB_Pcal_prior
      print *, 'meanPcal: ', meanPcal
      print *, 'sigmaPcal: ', sigmaPcal

      print *, 'Use kappa prior: ', PB_kappa_prior
      print *, 'meankappa: ', meankappa
      print *, 'sigmakappa: ', sigmakappa

      print *, 'Use alphaEE prior: ', PB_alphaEE_prior
      print *, 'meanAlphaEE: ', meanAlphaEE
      print *, 'sigmaAlphaEE: ', sigmaAlphaEE

   endif

   !Read in the bandpower and cov files, and determine where the windows are.
   bp_file = Ini%Read_String_Default('pb_EE_bp_file','')

   !I could simplify this by moving all the relevant info from
   !desc_file into the ini file.
   desc_file = Ini%Read_String_Default('pb_EE_desc_file', '')
   
   param_file = Ini%Read_String_Default( 'pb_EE_params_file','')
   if (feedback > 1) then
      print *,'Params file: ',param_file
      print*,'BP file: ',bp_file
      print *, 'desc file: ',desc_file
   endif
   call this%loadParamNames(param_file)




   !do we want extra debug prints
   printDlPB = Ini%Read_Logical('print_spectrum',.false.)
   if (printDlPB .and. MPIRank /= 0) then
      call MPIStop('Warning - print_spectrum is not MPI thread-safe, quitting...')
   endif



   !default to binary since faster i/o
   binaryCov=.True.
   binaryWindows=.True.
   binaryBeamErr=.True.
   cov_file =  Ini%Read_String_Default('pb_EE_binary_covariance','')
   if (cov_file == '') then 
      cov_file = Ini%Read_String_Default('pb_EE_covariance','')
      binaryCov=.False.
   endif
   window_folder = Ini%Read_String_Default('pb_EE_binary_window','')
   if (window_folder == '') then 
      window_folder = Ini%Read_String_Default('pb_EE_windir','')
      binaryWindows=.False.
   endif
   beamerr_file = Ini%Read_String_Default('pb_EE_binary_beamerr','')
   if (beamerr_file == '') then 
      beamerr_file = Ini%Read_String_Default('pb_EE_beamerr','')
      binaryBeamErr=.False.
   endif

   
   if (bp_file=='' .or. desc_file=='' .or. window_folder=='' .or. cov_file=='' .or. beamerr_file == '') then
      print*,'Missing required pb key: received: ',bp_file,desc_file,window_folder,cov_file,beamerr_file
      stop
   endif

   call this%InitPBData(desc_file, bp_file, cov_file, beamerr_file, window_folder)
 end subroutine PB_EE_ReadIni

 subroutine InitPBData(this, desc_file, bp_file, cov_file, beamerr_file, window_folder)
   !use IniFile
   implicit none
   class(TPolarbearEELike) :: this
   character(LEN=Ini_Enumeration_Len) :: desc_file, bp_file, cov_file,beamerr_file
   character(LEN=Ini_Enumeration_Len) :: window_folder
   integer, parameter :: tmp_file_unit=82
   integer i,j,k,l,dum,lwin,j0,j1
   integer*4 :: neff
   integer*8 :: offset,delta
   integer*4 :: efflmin,efflmax
   real*8 :: arr(2),rdum
   double precision, dimension(:), allocatable :: locwin,deltacb
   Type(TTextFile) :: F
   integer*4 :: errcode
   logical wexist

   !Obtain necessary info from the desc_file pertaining
   !to which freqs we're using, ell range, and number of bins per spectrum.
   call F%Open(desc_file)
   read(F%unit,*) nall !Number of bandpowers 
   read (F%unit,*) pb_windows_lmin, pb_windows_lmax !Min and Max ell in window file
   call F%Close()

   if (feedback > 1) then 
      print *, 'nbin: ', nall
      print *, 'pb_windows_lmin: ', pb_windows_lmin
      print *, 'pb_windows_lmax: ', pb_windows_lmax
      print *, 'window_folder: ', trim(window_folder)
   endif
   allocate(this%cl_lmax(CL_E,CL_E), source=0)
   this%cl_lmax(CL_E,CL_E) = pb_windows_lmax+1

   if (pb_windows_lmin < 2 .or. pb_windows_lmin >= pb_windows_lmax) then
      call mpistop('Invalid lranges for pb')
   end if

   !ells vector is 2 ell longer in order to do cl derivatives.
   !As a result, so is cl_to_dl_conversion
   allocate(ells(pb_windows_lmin-1:pb_windows_lmax+1), &
        cl_to_dl_conversion(pb_windows_lmin-1:pb_windows_lmax+1), &
        rawspec_factor(pb_windows_lmin-1:pb_windows_lmax+1), &
        deriv_factor(pb_windows_lmin:pb_windows_lmax))

   allocate(windows(pb_windows_lmin:pb_windows_lmax,nall), &
        spec(nall))

   allocate(cov(1:nall,1:nall), cov_w_beam(1:nall,1:nall),beam_err(nall,nall),cov_tmp(1:nall,1:nall))

   !Define an array with the l*(l+1)/2pi factor to convert to Dl from Cl.
   do j=pb_windows_lmin-1,pb_windows_lmax+1
      ells(j) = j
   enddo
   cl_to_dl_conversion(:) = (ells*(ells+1d0))/TWOPI
   rawspec_factor = ells**3 / cl_to_dl_conversion
   deriv_factor = 0.5d0 / ells(pb_windows_lmin:pb_windows_lmax)**2

   ! Read in bandpowers
   !Should be  EE-only from PB analysis.

   call F%Open(bp_file)
   do j=1,nall
      read (F%unit,*) rdum,spec(j)
   end do
   call F%close()


   
   ! Read in covariance
   !Should be TE, EE
   call OpenReadBinaryFile(cov_file,tmp_file_unit,nall*8_8)
   do i=1,nall
      read(tmp_file_unit,rec=i)cov(:,i)
   enddo
   close (tmp_file_unit)

   if (feedback > 1)   print *, 'First entry of covariance matrix: ', cov(1,1)

   if (feedback > 2) then
      print *,'Testing for negative evalues in PB matrix'
      cov_tmp = cov
      allocate(deltacb(nall))
      deltacb=0
      rdum = Matrix_GaussianLogLikeDouble(cov_tmp,deltacb)
      print *,'... no negative evalues in PB matrix'
      
   endif
   ! Read in windows
   if (binaryWindows) then
      inquire(FILE=trim(window_folder),EXIST=wexist)
      if (.not. wexist) then
         print*,'SPT hiell 2019, missing window file:', trim(window_folder)
         call mpistop()
      endif
      call OpenReadBinaryStreamFile(trim(window_folder),tmp_file_unit)
      read(tmp_file_unit,pos=1)efflmin,efflmax
      allocate(locwin(efflmin:efflmax))
      if ((efflmax .lt. pb_windows_lmin) .or. (efflmin .gt. pb_windows_lmax)) &
           call MpiStop('unallowed l-ranges for binary window functions')
      j0=efflmin
      if (pb_windows_lmin > j0) j0=pb_windows_lmin
      j1=efflmax
      if (pb_windows_lmax < j1) j1=pb_windows_lmax
      if (j1 < j0) &
           call MpiStop('unallowed l-ranges for binary window functions - no allowed ells')
      delta=(efflmax-efflmin+1)*8_8
      offset=2 * 4_8+1
      do i=1,nall
         read(tmp_file_unit,pos=((i-1)*delta + offset)) locwin
         windows(j0:j1,i)=locwin(j0:j1)
      end do
      close(tmp_file_unit)
      deallocate(locwin)
   else
      do i=1,nall
         inquire(FILE=trim(window_folder)//trim(numcat('window_',i)),EXIST=wexist)
         if (.not. wexist) then
            print*,'SPTpol, missing window file:', trim(window_folder)//trim(numcat('window_',i))
            call mpistop()
         endif
         call F%Open(trim(window_folder)//trim(numcat('window_',i)))
         do j=pb_windows_lmin,pb_windows_lmax
            read (F%unit,*) dum, windows(j,i)
         end do
         call F%Close()
      end do
   end if
      inquire(FILE=trim(beamerr_file),EXIST=wexist)
   if (.not. wexist) then
      print*,'SPT hiell 2019, missing beamerr file:', trim(beamerr_file)
      call mpistop()
   endif
   if (binaryBeamErr) then 
      call OpenReadBinaryFile(beamerr_file,tmp_file_unit,nall*8_8)
      do i=1,nall
         read(tmp_file_unit,rec=i)beam_err(:,i)
      enddo
      close (tmp_file_unit)
   else
      call F%open(beamerr_file)
      do i=1,nall
         do j=1,nall
            read (F%unit,*) beam_err(j,i)
         end do
      end do
      call F%close()
   endif
   if (feedback > 1) print *, 'First entry of beam correlation matrix: ', beam_err(1,1)
   
   SuccessfulPBEEInitialization = .true.

   if (feedback > 1) then
      print *, 'Successfully initialized PB_EE data...'
   endif

   print*,'Warning -- havent recalculated cos(dipole) terms for PB field. Using Henning et al value, which should be closeish. this is for aberration.'

 end subroutine InitPBData



 function PBEELnLike(this, CMB, Theory, DataParams) 
   implicit none
   class(TPolarbearEELike) :: this
   Class(CMBParams) :: CMB
   Class(TCosmoTheoryPredictions), target :: Theory
   double precision :: DataParams(:) 
   double precision, dimension(pb_windows_lmax+1) :: dls
   double precision :: PriorLnLike
   double precision :: dum
   double precision :: PBEELnLike
   double precision, parameter :: d3000 = 3000*3001/TWOPI
   double precision, parameter :: beta = 0.0012309
   double precision, parameter :: dipole_cosine = -0.4033
   double precision, dimension(1:nall) :: deltacb,BeamFac
   double precision, dimension(1:nall) :: tmpcb
   double precision, dimension(1) :: junk, detcov
   double precision :: PoissonLevel, ADust, alphaDust, CalFactor
   integer :: i,j,k, l,kk
   double precision :: norm
   integer fid
   real*4, dimension(2) :: arr
   real*4, dimension(7) :: arr7
   integer*4 :: errcode
   double precision :: lnl,beamlnl
   integer, dimension(1)::ivec
   double precision :: minlnl,loclnl

   double precision, dimension(pb_windows_lmin:pb_windows_lmax) :: dl_fgs
   double precision, dimension(pb_windows_lmin:pb_windows_lmax) :: cl_derivative
   double precision, dimension(pb_windows_lmin:pb_windows_lmax) :: aberration
   double precision, dimension(pb_windows_lmin-1:pb_windows_lmax+1) :: raw_spectra
   !DataParams for PB likelihood are: [kappa, Czero_EE, Adust_EE, alpha_EE,EEcal
   integer, parameter :: iKappa = 1,iPS_EE=2, iDust_EE=3, iDustAlpha_EE=4, iEEcal=5

   !first we need the cl arrays. this function returns Dls
   !so convert to Cls. We could eliminate this step by changing the later lines, but I was lazy
   call Theory%ClArray(dls,CL_E,CL_E)


   !First calculate Cl derivatives for this position in parameter space.
   !raw_spectra = l^3 C_l
   !CAMB theory curves are in Cl not Dl!!! So we don't need to do the dl_to_cl_conversion.
   !zero it; will be set if correct_aberrations is true
   aberration=0
   raw_spectra = rawspec_factor *  dls(pb_windows_lmin-1:pb_windows_lmax+1)

   !The derivative at point n is roughly (raw_spectra[n+1] - raw_spectra[n-1])/2.  Then we devide by l^2 for the proper scaling for the
   !kappa parameter as described in Manzotti, et al. 2014, equation (32).
   cl_derivative = deriv_factor * (raw_spectra(pb_windows_lmin+1:pb_windows_lmax+1) - raw_spectra(pb_windows_lmin-1:pb_windows_lmax-1))

   if (correct_aberration) then
      if (feedback > 1) then
         print *,'PB: Correcting for aberration...'
      end if
      !Also get derives of the Dls for use with aberration corrections.
      aberration = (dls(pb_windows_lmin+1:pb_windows_lmax+1) - dls(pb_windows_lmin-1:pb_windows_lmax-1))/2.
      aberration = (-1*beta*dipole_cosine)* ells(pb_windows_lmin:pb_windows_lmax)*aberration
   ENDIF

   !DataParams for PB likelihood are: [kappa, Czero_EE, Adust_EE, alpha_EE, EECalibrationFactor]

   PoissonLevel = DataParams(iPS_EE)/d3000 !TE/EE Poisson
   ADust = DataParams(iDust_EE) !EE dust amplitude, in Dl, NOT Cl yet...
   alphaDust = DataParams(iDustAlpha_EE) ! EE dust spectral index (for Dl).

   CalFactor = (DataParams(iEEcal))

   tmpcb(:)=0

   !First get model foreground spectrum (in Cl).
   !Note all the foregrounds are recorded in Dl at l=3000, so we 
   !divide by d3000 to get to a normalized Cl spectrum.
   
   !Start with Poisson power and subtract the kappa parameter for super sample lensing.
   dl_fgs = (PoissonLevel - DataParams(iKappa)*cl_derivative ) * cl_to_dl_conversion(pb_windows_lmin:pb_windows_lmax)

      !Now add model CMB.
   dl_fgs = dl_fgs + dls(pb_windows_lmin:pb_windows_lmax)
   !Do we want to correct for aberration?
   dl_fgs = dl_fgs + aberration

   ! add dust foreground model (defined in Dl)
   dl_fgs = dl_fgs + Adust*(ells(pb_windows_lmin:pb_windows_lmax)/80.0d0)**(alphaDust)

   if (printDlPB) then

      fid=33
      call OpenWriteBinaryFile(trim(numcat('like_tests/suxp_pb_',k)),fid,4_8 * 2)
      do l=pb_windows_lmin,pb_windows_lmax
         arr(1)=l
         arr(2)=dl_fgs(l)
         write(fid,rec=l-pb_windows_lmin+1) arr(1:2)
      enddo
      close(fid)

      call OpenWriteBinaryFile(trim(numcat('like_tests/suxp_pb_components_',k)),fid,4_8 * 7)
      do l=pb_windows_lmin,pb_windows_lmax
         arr7(1)=l
         arr7(2)=dl_fgs(l)
         arr7(3)=dls(l)
         arr7(4)=PoissonLevel*cl_to_dl_conversion(l)
         arr7(5)=DataParams(iKappa)*cl_derivative(l) *cl_to_dl_conversion(l)
         arr7(6)= Adust*(l/80d0)**(alphaDust+2.0d0)
         arr7(7)=aberration(l)
         write(fid,rec=l-pb_windows_lmin+1) arr7(1:7)
      enddo
      close(fid)
   endif

   !Now bin into bandpowers with the window functions.
   call dgemv('T',pb_windows_lmax-pb_windows_lmin+1,nall,1.0d0,&
        windows(:,1:nall),pb_windows_lmax-pb_windows_lmin+1,&
        dl_fgs,1,0d0,tmpcb,1)
   
   !scale theory spectrum by calibration:
   tmpcb = tmpcb(:) / CalFactor

   if (printDlPB) then
      fid=33
      open(fid,file=trim(numcat('like_tests/est_bandpowers_pb_',k)))
      do l=1,nall
         write(fid,*)l,tmpcb(l),spec(l)
      enddo
      close(fid)
   endif

   deltacb = tmpcb - spec

   do i=1,nall
      do j=1,nall
         cov_w_beam(i,j) = beam_err(i,j)*tmpcb(i)*tmpcb(j)
      enddo
   enddo

   cov_w_beam = cov + cov_w_beam
   if (feedback > 1) then
      cov_tmp=cov_w_beam
      detcov = Matrix_GaussianLogLikeDouble(cov_tmp,deltacb*0)
      print *, 'PBLnLike log(det(cov))/2 = ', detcov
   endif
   PBEELnLike =  Matrix_GaussianLogLikeDouble(cov_w_beam, deltacb)


   PriorLnLike = 0
   !Add Gaussian prior for polarization efficiency.
   if (PB_Pcal_prior) then
      PriorLnLike = PriorLnLike +   0.5d0*(log(CalFactor / meanPcal)/sigmaPcal)**2
      if (feedback > 1) print*,'Cal prior:',PriorLnLike
   endif

   !Add Gaussian prior for kappa.
   if (PB_kappa_prior) then
      PriorLnLike = PriorLnLike + 0.5d0*((DataParams(iKappa) - meankappa) /sigmakappa)**2
      if (feedback > 1) print*,'Plus kappa prior:',PriorLnLike
   endif

   !Add Gaussian prior for alphaEE.
   if (PB_alphaEE_prior) then
      PriorLnLike = PriorLnLike + 0.5d0*((DataParams(iDustAlpha_EE) - meanAlphaEE) /sigmaAlphaEE)**2
      if (feedback > 1) print*,'Plus dust alpah prior:',PriorLnLike
   endif

   !Add Gaussian prior for alphaEE.
   if (PB_dustEE_prior) then
      PriorLnLike = PriorLnLike + 0.5d0*((DataParams(iDust_EE) - meanDustEE) /sigmaDustEE)**2
      if (feedback > 1) print*,'Plus dust amp prior:',PriorLnLike
   endif

   if (feedback > 1) print *, 'Prior contribution to pb:',PriorLnLike

   PBEELnLike = PBEELnLike + PriorLnLike

   if (feedback > 1)  then
      print *, 'PBLnLike lnlike = ', PBEELnLike
      print *, 'PBLnLike chisq (after priors) = ', 2*(PBEELnLike-detcov)
      print *, 'PBLnLike chisq (before priors) = ', 2*(PBEELnLike-detcov-PriorLnLike)
   endif
 end function PBEELnLike


 subroutine OpenTxtFile(aname, aunit)

   character(LEN=*), intent(IN) :: aname
   integer, intent(in) :: aunit     
   open(unit=aunit,file=aname,form='formatted',status='old', action='read', err=500)
   return                                                                        
500 call MpiStop('File not found: '//trim(aname))                               
 end subroutine OpenTxtFile

 subroutine OpenReadBinaryStreamFile(aname,aunit)
   character(LEN=*), intent(IN) :: aname
   integer, intent(in) :: aunit
   open(unit=aunit,file=aname,form='unformatted',access='stream', err=500)
   ! be aware the data is in LE: ,convert='LITTLE_ENDIAN')
   return

500 call MpiStop('File not found: '//trim(aname))
 end subroutine OpenReadBinaryStreamFile

 subroutine OpenReadBinaryFile(aname,aunit,record_length)
   character(LEN=*), intent(IN) :: aname
   integer, intent(in) :: aunit
   integer*8,intent(in) :: record_length
   open(unit=aunit,file=aname,form='unformatted',access='direct',recl=record_length,  err=500)
   return

500 call MpiStop('File not found: '//trim(aname))
 end subroutine openReadBinaryFile

 subroutine OpenWriteBinaryFile(aname,aunit,record_length)
   character(LEN=*), intent(IN) :: aname
   integer, intent(in) :: aunit
   integer*8,intent(in) :: record_length
   open(unit=aunit,file=aname,form='unformatted',status='replace',access='direct',recl=record_length, err=500)
   return

500 call MpiStop('File not able to be written to: '//trim(aname))
 end subroutine OpenWriteBinaryFile

end module CMB_Polarbear_EE
