# SPT-3G 2-year delensed EE + optimal ϕϕ (MUSE) likelihood


This repository contains the bandpowers and systematics likelihood for the SPT-3G 2-year delensed EE + optimal ϕϕ analysis from [Ge et al. 2024]().

The likelihood is provided in both Python or Julia. The Python package provides a Cobaya likelihood plugin. 

![](banner.png)

## Python

**Install:**

```shell
pip install .
# or if you need likelihood gradients: 
pip install .[jax]
```
Note that the `pip` version requires `>=24.0`.

**Usage:**

```python
import muse3glike, jax # (jax optional, only for gradients)

spt = muse3glike.spt3g_2yr_delensed_ee_optimal_pp_muse()

x = spt.example_inputs() 

spt.loglike(x) # likelihood

jax.grad(spt.loglike)(x) # likelihood gradients

# plotting bandpowers
errorbar(
    spt.BPWF["ϕϕ"] @ spt.BPWF["ℓ"],
    spt.d[spt.axis["ϕϕ"]], 
    yerr = sqrt(diag(spt.Σ))[spt.axis["ϕϕ"]]
)
```

**With Cobaya:**

Install `muse3glike` as above, then in your Cobaya yaml:

```yaml
likelihood:
  muse3glike.cobaya.spt3g_2yr_delensed_ee_optimal_pp_muse: null
```

If you want to use just ϕϕ or EE you can specify the components:

```yaml
likelihood:
  muse3glike.cobaya.spt3g_2yr_delensed_ee_optimal_pp_muse:
    components: ["ϕϕ"] # or ["EE"]
```

Currently you can't explicitly sample the systematics parameters in Cobaya, i.e. you can't do e.g. `components: ["ϕϕ", "EE", "Acals"]` (you can do it manually from Python or Julia though). The systematics however are _always marginalized over_, even if they're not explicitly sampled.

## Julia

**Install:**

```shell
] add https://github.com/SouthPoleTelescope/muse-3g-like
```

**Usage:**

```julia
using Muse3GLike

spt = Muse3GLike.spt3g_2yr_delensed_ee_optimal_pp_muse(
    components = ("ϕϕ", "EE") # optional choice of what to explicity sample
)

x = Muse3GLike.example_inputs(spt)

Muse3GLike.loglike(spt, x) # likelihood

gradient(x -> Muse3GLike.loglike(spt, x), x) # likelihood gradients (w/ ForwardDiff,Zygote,..)

# plotting bandpowers
plot(
    spt.BPWF[:ϕϕ] * spt.BPWF[:ℓ],
    spt.d[:ϕϕ], 
    yerr = sqrt.(diag(spt.Σ[:ϕϕ]))
)
```


## Notes

* The likelihood is only Gaussian in the transformed bandpower space calculated by the code. Untransformed bandpowers & errors should only be used for plotting and visualization, not for statistical inference.
* The initialization function takes an optional keyword argument `components` to specify which subset of spectra and systematics parameters should be explicit arguments to the `loglike` function. Components which are not specified are still marginalized over. So e.g. a run with `components=("ϕϕ","EE")` and `components=("ϕϕ","EE","Acals")` will yield identical cosmology constraints, which _include_ calibration uncertainty `Acal`. The difference is that in the latter case `Acal` will be a parameter in the chain whose degeneracies with other quantities can be examined. The full list of possible components is: `("ϕϕ", "EE", "Acals", "ψpols", "βs", "βps", "ϵqs", "ϵus")`
* The likelihood assumes a uniform prior on input theory spectra between 0 and 2 times the Planck theory at every multipole, and returns NaN otherwise.
