
try:
    import jax.numpy as np
except ImportError:
    import numpy as np

import h5py
import re
import os.path as osp

DATA_DIR = osp.join(osp.abspath(osp.dirname(__file__)), "dat")


class spt3g_2yr_delensed_ee_optimal_pp_muse:

    def __init__(
        self, 
        filename = None,
        components = None
    ):
    
        if filename is None:
            filename = osp.join(DATA_DIR, "90_150_220.h5")
        
        self.components = components = components or ["ϕϕ", "EE"]

        with h5py.File(filename, 'r') as f:
        
            # "axis" is the Julia ComponentArray axis as a string like
            # "(ϕϕ=1:10, EE=11:50)" which gives the indices into the
            # parameter vector for various components. here, parse it
            # into a dictionary like {"ϕϕ": slice(0, 10), "EE":
            # slice(10, 50)}
            axis = '{' + f["axis"].asstr()[()][1:-1] + '}'
            axis = re.sub(r'(\d+):(\d+)', r'slice(\1-1,\2)', axis)
            axis = re.sub(r'(\w+) = ', r"'\1': ", axis)
            axis = axis.replace('ViewAxis', '').replace('ShapedAxis', '')
            self.axis = axis = {k:v if isinstance(v,slice) else v[0] for (k,v) in eval(axis).items()}
            self.todict = lambda vec: {k: vec[axis[k]] for k in components}
            self.tovec = lambda dct: np.concatenate([dct[k] for k in components])
            dim = max([v.stop for v in axis.values()])

            # preprocess, here drop last EE bin
            self.preprocess = lambda x: {
                k : x["EE"][:-1] if k=="EE" else x[k]
                for k in components
            }
            preprocess_jac = np.vstack([self.tovec(self.preprocess(self.todict(np.eye(dim)[i,:]))) for i in range(dim)]).T

            # bandpower likelihood is taken as Gaussian in this transformed space. volume factor keeps uniform prior on bandpowers.
            self.transform = lambda x: {
                k : bandpower_transform(x[k]) if isbandpowers(k) else x[k]
                for k in components
            }
            transform_volume = np.diag(self.tovec({
                k : (1/0.8**2 if isbandpowers(k) else 0) * self.preprocess(self.todict(np.ones(dim)))[k]
                for k in components
            }))

            # preprocess and transform the data and covariance matrix
            s_full = f["scale"][:]
            self.s = self.todict(s_full)
            self.d = self.todict(f["data"][:])
            self.d_transformed = self.transform(self.preprocess({k : self.d[k] / self.s[k] for k in components}))
            self.d_transformed_vec = self.tovec(self.d_transformed)
            self.Σ = f["covariance"][:]
            self.Σ_transformed = np.linalg.inv(np.linalg.inv((preprocess_jac @ ((self.Σ / s_full).T / s_full) @ preprocess_jac.T)) + transform_volume)
            self.Σ_transformed_cho = np.linalg.cholesky(self.Σ_transformed)

            # bandpower window functions
            self.BPWF = {k: np.array(f["BPWF"][k][:]).T for k in ["ϕϕ", "EE"]}
            self.BPWF["ℓ"] = np.arange(1, self.BPWF["EE"].shape[1]+1)


    def loglike(self, x):
        μ = {
            k : (self.BPWF[k] @ x[k]) / self.s[k] if isbandpowers(k) else x[k]
            for k in self.components
        }
        Δ = self.d_transformed_vec - self.tovec(self.transform(self.preprocess(μ)))
        z = np.linalg.solve(self.Σ_transformed_cho, Δ)
        return -np.dot(z, z) / 2

    def example_inputs(self):
        with h5py.File(osp.join(DATA_DIR, "example_inputs.h5"), 'r') as f:
            x = {
                k: np.array(f[k][:]) if isbandpowers(k) else np.ones(self.d[k].size) if k=="Acals" else np.zeros(self.d[k].size)
                for k in self.components
            }
        return x
        

def isbandpowers(k):
    return k in ["ϕϕ", "EE"]

def bandpower_transform(x):
    return np.arctanh(x - 1.01)


